/* Copyright (C) 2011-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "VNCDeviceProviderBase.h"
#ifdef VNC_QNX
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#else
#include <cstring>
#include <cstdarg>
#include <cassert>
#endif
#ifdef _WIN32_WCE
#define vsnprintf _vsnprintf
#endif
#include <new>
#include <StringGenerator.h>
#include <StringParser.h>
#include <DiscoveryException.h>

VNCDeviceProviderImpl::VNCDeviceProviderImpl(VNCDeviceProviderInterface *pInterface, 
            size_t interfaceSize,
            VNCDeviceProviderCallbacks *pCallbacks,
            size_t callbacksSize)
  : mvpContext(NULL),
    mState(StateStopped),
    mSinglePhaseStart(false)
{
  VNCVerifyIntegerTypes();
  memset(&mCallbacks, 0, sizeof(mCallbacks));
  memcpy(&mCallbacks, 
        pCallbacks, 
        callbacksSize < sizeof(mCallbacks) ? 
                    callbacksSize : sizeof(mCallbacks));
  #undef IMPLEMENT
  #define IMPLEMENT(name) \
    if(offsetof(VNCDeviceProviderInterface, name) + sizeof(void*) <= \
                    interfaceSize) \
        pInterface->name = &VNCDeviceProviderImpl::name;
  IMPLEMENT(setContext);
  IMPLEMENT(setProperty);
  IMPLEMENT(getProperty);
  IMPLEMENT(start);
  IMPLEMENT(getEventHandlesEx);
  IMPLEMENT(handleEvent);
  IMPLEMENT(handleTimeout);
  IMPLEMENT(stop);
  IMPLEMENT(destroy);
  IMPLEMENT(getDeviceProperties);
  IMPLEMENT(startMonitoring);
  IMPLEMENT(startReporting);
  IMPLEMENT(stopMonitoring);
  IMPLEMENT(stopReporting);

  memset(&mUtils, 0, sizeof(VNCDiscoverySDKUtils));
}

VNCDeviceProviderImpl::~VNCDeviceProviderImpl()
{
}

// Logging methods
bool VNCDeviceProviderImpl::isSeverityLogged(VNCDiscovererLogSeverity severity)
{
  return ((*mCallbacks.isSeverityLogged)(mvpContext, severity) != 0);
}

void VNCDeviceProviderImpl::log(VNCDiscovererLogSeverity severity, 
                              const char* pText)
{
  if(!isSeverityLogged(severity))
    return;

  (*mCallbacks.log)(mvpContext, severity, pText);
}

void VNCDeviceProviderImpl::logf(VNCDiscovererLogSeverity severity, 
                              const char* pText, ...)
{
  if(!isSeverityLogged(severity))
    return;

  va_list va;
  va_start(va, pText);
  logv(severity, pText, va);
  va_end(va);
}

void VNCDeviceProviderImpl::logv(VNCDiscovererLogSeverity severity, 
                              const char* pText, va_list args)
{
  const int bufSize = 512;
  char buf[bufSize];
  vsnprintf(buf, bufSize - 1,
            pText, args);
  // Make sure it's NUL terminated
  buf[bufSize - 1] = '\0';
  log(severity, buf);
}

// Implementation of the interface
void VNCDeviceProviderImpl::setContext(void* pContext)
{
  mvpContext = pContext;
  memset(&mUtils, 0, sizeof(VNCDiscoverySDKUtils));
  if(mCallbacks.getUtils)
  {
    (*mCallbacks.getUtils)(mvpContext, &mUtils, sizeof(VNCDiscoverySDKUtils));
  }
}

void VNCDeviceProviderImpl::start()
{
  // This is the old API call. If the SDK calls start it means that it won't
  // call startMonitoring and startReporting. Call startMonitoring now, and if
  // that is successful call startReporting in the notifyStarted() callback.
  mSinglePhaseStart = true;
  if(getState() == StateStopped)
  {
    VNCDiscoverySDKError err = startMonitoring();
    if(err == VNCDiscoverySDKErrorNone)
    {
      mState = StateMonitoring;
    }
    else
    {
      notifyStarted(err);
    }
  }
  if(getState() == StateMonitoring)
  {
    startReporting();
  }
}

void VNCDeviceProviderImpl::stop()
{
  // This is the old API call. If the SDK calls stop it means that it won't
  // call stopMonitoring and stopReporting.
  mSinglePhaseStart = true;
  if(getState() == StateReporting)
  {
    stopReporting();
    return;
  }
  if(getState() == StateMonitoring)
  {
    mState = StateStopped;
    (*mCallbacks.notifyStopped)(mvpContext, stopMonitoring());
  }
}

VNCDiscoverySDKError VNCDeviceProviderImpl::setProperty(
    const char* /*pProperty*/, const char* /*pValue*/)
{
  return VNCDiscoverySDKErrorNotSupported;
}

VNCDiscoverySDKError VNCDeviceProviderImpl::getProperty(
    const char* /*pProperty*/, char** /*pValue*/)
{
  return VNCDiscoverySDKErrorNotSupported;
}

VNCDiscovererEventHandle *VNCDeviceProviderImpl::getEventHandlesEx(
    int **ppWriteNotification)
{
  try
  {
    mEventHandles.update();
  }
  catch(const std::bad_alloc&)
  {
    LOG_CRITICAL_ERROR(this, "Unable to prepare the multiple handles, "
        "out of memory");
  }
  *ppWriteNotification = mEventHandles.writeNotifications();
  return mEventHandles.handles();
}

void VNCDeviceProviderImpl::handleEvent(VNCDiscovererEventHandle* /*pHandle*/)
{
  assert(false);
}

void VNCDeviceProviderImpl::handleTimeout(VNCDiscovererEventHandle* /*pHandle*/)
{
  assert(false);
}

// Callbacks
void VNCDeviceProviderImpl::notifyStarted(VNCDiscoverySDKError error)
{
  if(error == VNCDiscoverySDKErrorNone)
  {
    mState = StateReporting;
  }
  else if(mSinglePhaseStart)
  {
    stopMonitoring();
    mState = StateStopped;
  }
  (*mCallbacks.notifyStarted)(mvpContext, error);
}

void VNCDeviceProviderImpl::notifyStopped(VNCDiscoverySDKError error)
{
  mState = StateMonitoring;
  if(mSinglePhaseStart)
  {
    mState = StateStopped;
    if(error == VNCDiscoverySDKErrorNone)
    {
      error = stopMonitoring();
    }
    else
    {
      stopMonitoring();
    }
  }
  (*mCallbacks.notifyStopped)(mvpContext, error);
}

void VNCDeviceProviderImpl::setTimer(VNCDiscovererEventHandle *pEventHandle,
    VNCDiscoverySDKTimeoutMicroseconds timeout)
{
  (*mCallbacks.setTimer)(mvpContext, pEventHandle, timeout);
}

void VNCDeviceProviderImpl::eventHandlesChanged()
{
  (*mCallbacks.eventHandlesChanged)(mvpContext);
}

void VNCDeviceProviderImpl::notifyDeviceFound(VNCDeviceInfo *pDevice)
{
  (*mCallbacks.notifyDeviceFound)(mvpContext, pDevice);
}

void VNCDeviceProviderImpl::notifyDeviceRemoved(vnc_uint32_t deviceId)
{
  (*mCallbacks.notifyDeviceRemoved)(mvpContext, deviceId);
}

void VNCDeviceProviderImpl::notifyDeviceRemovedFromAddress(const char *pAddress)
{
  (*mCallbacks.notifyDeviceRemovedFromAddress)(mvpContext, pAddress);
}

// Event handle helper methods
void VNCDeviceProviderImpl::addEventHandle(VNCDiscovererEventHandle *handleToAdd,
    bool notifySDK, bool writeNotification)
{
  if(!handleToAdd)
    return;
  mEventHandles.addHandle(handleToAdd, writeNotification);
  if(notifySDK)
  {
    eventHandlesChanged();
  }
}

void VNCDeviceProviderImpl::removeEventHandle(
    VNCDiscovererEventHandle *handleToRemove,
    bool notifySDK)
{
  if(!handleToRemove)
    return;
  mEventHandles.removeHandle(handleToRemove);
  if(notifySDK)
  {
    eventHandlesChanged();
  }
}

// Memory allocating/free
VNCDeviceInfo* VNCDeviceProviderImpl::allocDevice(vnc_uint32_t propertiesCount,
    vnc_uint32_t interfacesCount)
{
  return (*mCallbacks.allocDevice)(propertiesCount, interfacesCount);
}

void VNCDeviceProviderImpl::freeDevice(VNCDeviceInfo *pDeviceInfo)
{
  (*mCallbacks.freeDevice)(pDeviceInfo);
}

VNCDeviceInterface* VNCDeviceProviderImpl::allocDeviceInterface(vnc_uint32_t propertiesCount,
    vnc_uint32_t subInterfacesCount)
{
  return (*mCallbacks.allocDeviceInterface)(propertiesCount,
      subInterfacesCount);
}

void VNCDeviceProviderImpl::freeDeviceInterface(VNCDeviceInterface *pDeviceInterface)
{
  (*mCallbacks.freeDeviceInterface)(pDeviceInterface);
}

char* VNCDeviceProviderImpl::allocString(const char* pString, size_t length)
{
  return (*mCallbacks.allocString)(pString, length);
}

void VNCDeviceProviderImpl::freeString(char* pString)
{
  (*mCallbacks.freeString)(pString);
}

VNCDiscoverySDKKeyValPair* VNCDeviceProviderImpl::allocKeyValPairArray(size_t size)
{
  return (*mCallbacks.allocKeyValPairArray)(size);
}

void VNCDeviceProviderImpl::freeArray(VNCDiscoverySDKKeyValPair *pArray,
    size_t arraySize)
{
  (*mCallbacks.freeArray)(pArray, arraySize);
}

vncdiscovery::StringGenerator* VNCDeviceProviderImpl::createStringGenerator(
    VNCDiscoverySDKError *pError)
{
  VNCDiscoverySDKUtils* pUtils = getUtils();
  VNCDiscoverySDKError err = VNCDiscoverySDKErrorNone;
  vncdiscovery::StringGenerator* pGenerator = NULL;
  try
  {
    pGenerator = new vncdiscovery::StringGenerator(pUtils);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(this, "%s: %d (%s)", e.what(), e.errorCode,
        errorToString(e.errorCode));
    err = e.errorCode;
  }
  catch(const std::bad_alloc&)
  {
    LOG_ERROR(this, "Unable to create Generator: out of memory");
    err = VNCDiscoverySDKErrorOutOfMemory;
  }
  if(pError)
  {
    *pError = err;
  }
  return pGenerator;
}

vncdiscovery::StringParser* VNCDeviceProviderImpl::createStringParser(const char* string,
    VNCDiscoverySDKError *pError)
{
  VNCDiscoverySDKUtils* pUtils = getUtils();
  VNCDiscoverySDKError err = VNCDiscoverySDKErrorNone;
  vncdiscovery::StringParser* pParser = NULL;
  try
  {
    pParser = new vncdiscovery::StringParser(pUtils, string);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(this, "%s: %d (%s)", e.what(), e.errorCode,
        errorToString(e.errorCode));
    err = e.errorCode;
  }
  catch(const std::bad_alloc&)
  {
    LOG_ERROR(this, "Unable to create Parser: out of memory");
    err = VNCDiscoverySDKErrorOutOfMemory;
  }
  if(pError)
  {
    *pError = err;
  }
  return pParser;
}

const char* VNCDeviceProviderImpl::errorToString(VNCDiscoverySDKError error)
{
  if(!mCallbacks.errorToString)
  {
    return "<unknown>";
  }
  return (*mCallbacks.errorToString)(error);
}

// static API functions
void VNCCALL VNCDeviceProviderImpl::setContext(VNCDeviceProvider* pDeviceProvider,
    void* vpContext)
{
  // No throws are expected from this.
  pDeviceProvider->setContext(vpContext);
}

VNCDiscoverySDKError VNCCALL VNCDeviceProviderImpl::setProperty(
    VNCDeviceProvider* pDeviceProvider,
    const char* pProperty,
    const char* pValue)
{
  try
  {
    return pDeviceProvider->setProperty(pProperty, pValue);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    return e.errorCode;
  }
  catch(const std::bad_alloc& /*e*/)
  {
    LOG_ERROR(pDeviceProvider, "VNCDeviceProvider::setProperty failed, "
        "out of memory");
    return VNCDiscoverySDKErrorOutOfMemory;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::setProperty "
        "caught exception: %s", e.what());
    (void) e;
  }
  return VNCDiscoverySDKFailed;
}

VNCDiscoverySDKError VNCCALL VNCDeviceProviderImpl::getProperty(
    VNCDeviceProvider *pDeviceProvider,
    const char *pProperty,
    char **ppValue)
{
  try
  {
    return pDeviceProvider->getProperty(pProperty, ppValue);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    return e.errorCode;
  }
  catch(const std::bad_alloc& /*e*/)
  {
    LOG_ERROR(pDeviceProvider, "VNCDeviceProvider::getProperty failed, "
        "out of memory");
    return VNCDiscoverySDKErrorOutOfMemory;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::getProperty "
        "caught exception: %s", e.what());
    (void) e;
  }
  return VNCDiscoverySDKFailed;
}

void VNCCALL VNCDeviceProviderImpl::start(VNCDeviceProvider *pDeviceProvider)
{
  try
  {
    pDeviceProvider->start();
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    pDeviceProvider->notifyStarted(e.errorCode);
  }
  catch(const std::bad_alloc& /*e*/)
  {
    LOG_ERROR(pDeviceProvider, "VNCDeviceProvider::start failed, "
        "out of memory");
    pDeviceProvider->notifyStarted(VNCDiscoverySDKErrorOutOfMemory);
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::start "
        "caught exception: %s", e.what());
    pDeviceProvider->notifyStarted(VNCDiscoverySDKFailed);
    (void) e;
  }
}

VNCDiscovererEventHandle *VNCCALL VNCDeviceProviderImpl::getEventHandlesEx(
    VNCDeviceProvider *pDeviceProvider,
    int **ppWriteNotification)
{
  // No throws are expected from this.
  return pDeviceProvider->getEventHandlesEx(ppWriteNotification);
}

void VNCCALL VNCDeviceProviderImpl::handleEvent(VNCDeviceProvider *pDeviceProvider,
    VNCDiscovererEventHandle* pHandle)
{
  try
  {
    pDeviceProvider->handleEvent(pHandle);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    (void) e;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::handleEvent "
        "caught exception: %s", e.what());
    (void) e;
  }
}

void VNCCALL VNCDeviceProviderImpl::handleTimeout(VNCDeviceProvider *pDeviceProvider,
    VNCDiscovererEventHandle* pHandle)
{
  try
  {
    pDeviceProvider->handleTimeout(pHandle);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    (void) e;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::handleEvent "
        "caught exception: %s", e.what());
    (void) e;
  }
}

void VNCCALL VNCDeviceProviderImpl::stop(VNCDeviceProvider *pDeviceProvider)
{
  try
  {
    pDeviceProvider->stop();
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    pDeviceProvider->notifyStopped(e.errorCode);
  }
  catch(const std::bad_alloc& /*e*/)
  {
    LOG_ERROR(pDeviceProvider, "VNCDeviceProvider::stop failed, "
        "out of memory");
    pDeviceProvider->notifyStopped(VNCDiscoverySDKErrorOutOfMemory);
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::stop "
        "caught exception: %s", e.what());
    pDeviceProvider->notifyStopped(VNCDiscoverySDKFailed);
    (void) e;
  }
}

void VNCCALL VNCDeviceProviderImpl::destroy(VNCDeviceProvider *pDeviceProvider)
{
  // No throws are expected from this.
  delete pDeviceProvider;
}

size_t VNCCALL VNCDeviceProviderImpl::getDeviceProperties(VNCDeviceProvider *pDeviceProvider,
      const VNCDeviceInfo *pDevice,
      VNCDiscoverySDKKeyValPair **ppProperties)
{
  try
  {
    return pDeviceProvider->getDeviceProperties(pDevice, ppProperties);
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    (void) e;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::getDeviceProperties "
        "caught exception: %s", e.what());
    (void) e;
  }
  return 0;
}

VNCDiscoverySDKError VNCCALL VNCDeviceProviderImpl::startMonitoring(VNCDeviceProvider *pDeviceProvider)
{
  try
  {
    if(pDeviceProvider->getState() == StateStopped)
    {
      VNCDiscoverySDKError err = pDeviceProvider->startMonitoring();
      if(err == VNCDiscoverySDKErrorNone)
      {
        pDeviceProvider->mState = StateMonitoring;
      }
      else
      {
        LOG_DEBUG_F(pDeviceProvider, "VNCDeviceProvider::startMonitoring "
            "failed with error %d (%s)", err,
            pDeviceProvider->errorToString(err));
      }
      return err;
    }
    else
    {
      LOG_WARNING(pDeviceProvider, "VNCDeviceProvider::startMonitoring "
          "called when the device provider is not stopped");
      return VNCDiscoverySDKErrorIllegalWhileRunning;
    }
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    return e.errorCode;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::startMonitoring "
        "caught exception: %s", e.what());
    (void) e;
  }
  return VNCDiscoverySDKErrorUnableToInitialize;
}

void VNCCALL VNCDeviceProviderImpl::startReporting(VNCDeviceProvider *pDeviceProvider)
{
  try
  {
    if(pDeviceProvider->getState() == StateMonitoring)
    {
      pDeviceProvider->startReporting();
    }
    else
    {
      LOG_WARNING(pDeviceProvider, "VNCDeviceProvider::startReporting "
          "called when the device provider is not monitoring");
    }
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    pDeviceProvider->notifyStarted(e.errorCode);
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::startReporting "
        "caught exception: %s", e.what());
    (void) e;
    pDeviceProvider->notifyStarted(VNCDiscoverySDKErrorUnableToInitialize);
  }
}

VNCDiscoverySDKError VNCCALL VNCDeviceProviderImpl::stopMonitoring(VNCDeviceProvider *pDeviceProvider)
{
  try
  {
    if(pDeviceProvider->getState() == StateMonitoring)
    {
      VNCDiscoverySDKError err = pDeviceProvider->stopMonitoring();
      pDeviceProvider->mState = StateStopped;
      return err;
    }
    else
    {
      LOG_WARNING(pDeviceProvider, "VNCDeviceProvider::stopMonitoring "
          "called when the device provider is not monitoring");
      return VNCDiscoverySDKErrorIllegalWhileNotRunning;
    }
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    return e.errorCode;
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::stopMonitoring "
        "caught exception: %s", e.what());
    (void) e;
  }
  return VNCDiscoverySDKFailed;
}

void VNCCALL VNCDeviceProviderImpl::stopReporting(VNCDeviceProvider *pDeviceProvider)
{
  try
  {
    if(pDeviceProvider->getState() == StateReporting)
    {
      pDeviceProvider->stopReporting();
    }
    else
    {
      LOG_WARNING(pDeviceProvider, "VNCDeviceProvider::stopReporting "
          "called when the device provider is not reporting");
    }
  }
  catch(vncdiscovery::DiscoveryException& e)
  {
    LOG_ERROR_F(pDeviceProvider, "%s: %d (%s)", e.what(), e.errorCode,
        pDeviceProvider->errorToString(e.errorCode));
    pDeviceProvider->notifyStopped(e.errorCode);
  }
  catch(const std::exception& e)
  {
    LOG_ERROR_F(pDeviceProvider, "VNCDeviceProvider::stopReporting "
        "caught exception: %s", e.what());
    (void) e;
    pDeviceProvider->notifyStopped(VNCDiscoverySDKFailed);
  }
}

VNCDiscoverySDKUtils* VNCDeviceProviderImpl::getUtils()
{
  return &mUtils;
}

VNCDeviceProviderImpl::State VNCDeviceProviderImpl::getState() const
{
  return mState;
}

VNCDeviceProviderImpl::DeviceAutoPtr::DeviceAutoPtr(
    VNCDeviceProviderImpl& devProvider,
    VNCDeviceInfo* pDevice)
  : mDevProvider(devProvider),
    mpDevice(pDevice)
{ }

VNCDeviceProviderImpl::DeviceAutoPtr::~DeviceAutoPtr()
{
  reset();
}

void VNCDeviceProviderImpl::DeviceAutoPtr::reset(
    VNCDeviceInfo* pDevice)
{
  if(mpDevice)
  {
    mDevProvider.freeDevice(mpDevice);
  }
  mpDevice = pDevice;
}

VNCDeviceInfo* VNCDeviceProviderImpl::DeviceAutoPtr::release()
{
  VNCDeviceInfo* pDevice = mpDevice;
  mpDevice = NULL;
  return pDevice;
}

VNCDeviceInfo* VNCDeviceProviderImpl::DeviceAutoPtr::get()
{
  return mpDevice;
}

VNCDeviceInfo* VNCDeviceProviderImpl::DeviceAutoPtr::operator->()
{
  return mpDevice;
}

VNCDeviceProviderImpl::DeviceAutoPtr::operator bool() const
{
  return (mpDevice != NULL);
}
